import { AuthenticationError, PromiseReturnType } from "blitz";
import Link from "next/link";
import { LabeledTextField } from "app/core/components/LabeledTextField/LabeledTextField";
import { Form, FORM_ERROR } from "app/core/components/Form/Form";
import login from "app/auth/mutations/login";
import { Login } from "app/auth/validations";
import { useMutation } from "@blitzjs/rpc";
import { Routes } from "@blitzjs/next";
import Header from "app/components/Header/Header";
import styles from "./LoginForm.module.css";

type LoginFormProps = {
  onSuccess?: (user: PromiseReturnType<typeof login>) => void;
};

export const LoginForm = (props: LoginFormProps) => {
  const [loginMutation] = useMutation(login);
  return (
    <div>
      <Header>Login</Header>

      <Form
        submitText="Login"
        schema={Login}
        initialValues={{ email: "", password: "" }}
        onSubmit={async (values) => {
          try {
            const user = await loginMutation(values);
            props.onSuccess?.(user);
          } catch (error: any) {
            if (error instanceof AuthenticationError) {
              return { [FORM_ERROR]: "Sorry, those credentials are invalid" };
            } else {
              return {
                [FORM_ERROR]:
                  "Sorry, we had an unexpected error. Please try again. - " + error.toString(),
              };
            }
          }
        }}
      >
        <LabeledTextField name="email" label="Email" placeholder="Email" />
        <LabeledTextField name="password" label="Password" placeholder="Password" type="password" />
      </Form>
      <section className={styles.links}>
        <div>
          <Link href={Routes.ForgotPasswordPage()}>
            <a>Forgot your password?</a>
          </Link>
        </div>
        <div style={{ marginTop: "1rem" }}>
          Or{" "}
          <Link href={Routes.SignupPage()}>
            <a>Sign Up</a>
          </Link>
        </div>
      </section>
    </div>
  );
};

export default LoginForm;
