import { forwardRef, PropsWithoutRef } from "react";
import { useField, UseFieldConfig } from "react-final-form";

export interface LabeledTextFieldProps extends PropsWithoutRef<JSX.IntrinsicElements["input"]> {
  /** Field name. */
  name: string;
  /** Field type. Doesn't include radio buttons and checkboxes */
  type?: "text" | "password" | "email" | "number";
  outerProps?: PropsWithoutRef<JSX.IntrinsicElements["div"]>;
  fieldProps?: UseFieldConfig<string>;
}

export const TextField = forwardRef<HTMLInputElement, LabeledTextFieldProps>(
  ({ name, outerProps, fieldProps, ...props }, ref) => {
    const {
      input,
      meta: { touched, error, submitError, submitting },
    } = useField(name, {
      parse:
        props.type === "number"
          ? (Number as any)
          : // Converting `""` to `null` ensures empty values will be set to null in the DB
            (v) => (v === "" ? null : v),
      ...fieldProps,
    });

    const normalizedError = Array.isArray(error) ? error.join(", ") : error || submitError;

    return (
      <div {...outerProps}>
        <input {...input} disabled={submitting} {...props} ref={ref} />

        {touched && normalizedError && (
          <div role="alert" style={{ color: "red" }}>
            {normalizedError}
          </div>
        )}
      </div>
    );
  }
);

export default TextField;
