import { useEffect } from "react";
import { useRouter } from "next/router";
import Link from "next/link";
import { Routes } from "@blitzjs/next";
import { usePaginatedQuery } from "@blitzjs/rpc";
import getShortUrls from "app/short-urls/queries/getShortUrls";
import Header from "app/components/Header/Header";
import ShortUrlsList from "app/short-urls/components/ShortUrlsList/ShortUrlsList";
import ShortUrlListItem from "app/short-urls/components/ShortUrlListItem/ShortUrlListItem";
import ShortUrlLink from "../ShortUrlLink/ShortUrlLink";

const ITEMS_PER_PAGE = 100;

const ShortUrlsSection = ({ lastShortUrlId }) => {
  const router = useRouter();
  const page = Number(router.query.page) || 0;
  const [{ shortUrls, hasMore }, { refetch }] = usePaginatedQuery(getShortUrls, {
    skip: ITEMS_PER_PAGE * page,
    take: ITEMS_PER_PAGE,
    orderBy: {
      createdAt: "desc",
    },
  });
  useEffect(() => {
    void refetch();
  }, [lastShortUrlId, refetch]);

  const goToPreviousPage = () => router.push({ query: { page: page - 1 } });
  const goToNextPage = () => router.push({ query: { page: page + 1 } });

  if (shortUrls.length === 0) {
    return <section>
      <Header>No URLs shortened yet</Header>
    </section>
  }

  return (
    <section>
      <Header>Your URLs</Header>
      <ShortUrlsList>
        {shortUrls.map((shortUrl) => (
          <ShortUrlListItem key={shortUrl.id} isActive={shortUrl.id === lastShortUrlId}>
            <td>
              <kbd>{shortUrl.url}</kbd>
              <br />
              <kbd>
                <ShortUrlLink id={shortUrl.id} />
              </kbd>
            </td>
            <td>
              <Link href={Routes.ManageShortUrlPage({ id: shortUrl.id })}>stats & management</Link>
            </td>
          </ShortUrlListItem>
        ))}
      </ShortUrlsList>

      {(hasMore || page !== 0) && (
        <>
          <button disabled={page === 0} onClick={goToPreviousPage}>
            Previous
          </button>
          <button disabled={!hasMore} onClick={goToNextPage}>
            Next
          </button>
        </>
      )}
    </section>
  );
};

export default ShortUrlsSection;
