import { resolver } from "@blitzjs/rpc";
import db from "db";
import { CreateShortUrl } from "app/auth/validations";
import { Ctx } from "blitz";
import z from "zod";
import generateUniqueShortUrlId from "app/short-urls/domain/generateUniqueShortUrlId";

export default resolver.pipe(
  resolver.zod(CreateShortUrl),
  resolver.authorize(),
  async (input: z.infer<typeof CreateShortUrl>, ctx: Ctx) => {
    const url = input.url;
    const userId = ctx.session.userId;
    if (userId == null) {
      return;
    }

    const existingShortUrl = await db.shortUrl.findUnique({ where: { url_userId: { url, userId } } });
    if (existingShortUrl != null) {
      return existingShortUrl;
    }

    const checkIdIsUsed = (id) => db.shortUrl.findUnique({ where: { id } }).then(Boolean);
    const id = await generateUniqueShortUrlId(`${userId}-${url}`, checkIdIsUsed);

    const shortUrl = await db.shortUrl.create({ data: { id, url, userId } })
    if (!!process.env.BACKFILL_HIT_COUNTER) {
      let backFillDays = 5;
      let date = new Date();

      while (backFillDays > 0) {
        date.setDate(date.getDate() - 1);
        await db.hitCounter.create({
          data: {
            date: date.toISOString().slice(0, 10),
            shortUrlId: id,
            count: Math.round(Math.random() * 8),
          },
        });
        backFillDays--;
      }
    }
    return shortUrl;
  }
);
